import React, { useState, useEffect, useMemo } from 'react';
import { useNavigate } from 'react-router-dom';
import { Container, Row, Col, Pagination, Alert, Form, Button, Spinner } from 'react-bootstrap';
import { FaSearch, FaExclamationCircle } from 'react-icons/fa';
import JobCard from '../../components/faculty/JobCard';
import './styles/JobOpportunities.css';

const normalize = (str = '') => str.toString().toLowerCase().trim();

// Sample job data
const sampleJobs = [
  {
    id: 1,
    title: 'Computer Science Professor',
    department: 'Computer Science',
    job_type: 'Full-time',
    location: 'New York, NY',
    salary: '$80,000 - $100,000',
    deadline: '2023-12-31',
    course: 'Computer Science',
    postedAt: '2023-10-15T10:00:00Z',
    description: 'We are looking for an experienced Computer Science professor...',
    requirements: 'PhD in Computer Science or related field...',
    responsibilities: 'Teach undergraduate and graduate courses...'
  },
  // Add more sample jobs as needed
];

const JobOpportunities = () => {
  const navigate = useNavigate();
  const [currentPage, setCurrentPage] = useState(1);
  const [sortOption, setSortOption] = useState('newest');
  const [savedJobs, setSavedJobs] = useState([]);
  const [appliedJobs, setAppliedJobs] = useState([]);
  const [showSaveAlert, setShowSaveAlert] = useState(false);
  const [lastSavedJob, setLastSavedJob] = useState(null);
  const [searchQuery, setSearchQuery] = useState('');
  const [debouncedQuery, setDebouncedQuery] = useState('');
  const [jobs, setJobs] = useState(sampleJobs);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const pageSize = 6;

  /* Debounce search input to avoid filtering on every keystroke */
  useEffect(() => {
    const t = setTimeout(() => setDebouncedQuery(searchQuery), 300);
    return () => clearTimeout(t);
  }, [searchQuery]);

  // When sort or debounced query changes reset to page 1
  useEffect(() => {
    setCurrentPage(1);
  }, [sortOption, debouncedQuery]);

  // Load saved jobs from localStorage on mount
  useEffect(() => {
    const saved = JSON.parse(localStorage.getItem('savedJobs') || '[]');
    setSavedJobs(saved);
  }, []);

  // Scroll to top when component mounts
  useEffect(() => {
    window.scrollTo(0, 0);
  }, []);

  const handleSaveJob = (job, isSaved) => {
    let updatedSavedJobs;

    if (isSaved) {
      updatedSavedJobs = [...savedJobs, job];
      setLastSavedJob(job);
      setShowSaveAlert(true);
      setTimeout(() => setShowSaveAlert(false), 3000);
    } else {
      updatedSavedJobs = savedJobs.filter(savedJob => savedJob.id !== job.id);
    }

    setSavedJobs(updatedSavedJobs);
    localStorage.setItem('savedJobs', JSON.stringify(updatedSavedJobs));
  };

  const isJobSaved = (jobId) => savedJobs.some(job => job.id === jobId);

  const handleApply = (jobId) => {
    if (!appliedJobs.includes(jobId)) {
      setAppliedJobs(prev => [...prev, jobId]);
      alert('Application submitted successfully!');
    } else {
      alert('You have already applied to this position.');
    }
  };

  /* Search and sort logic:
     - Tokenizes the query ("can acad" => ["can","acad"])
     - Checks that every token exists in at least one searchable field (AND behavior)
     - Fields normalized to lowercase for case-insensitive match
  */
  const filteredJobs = useMemo(() => {
    let filtered = [...jobs];

    const q = normalize(debouncedQuery);
    const tokens = q ? q.split(/\s+/).filter(Boolean) : [];

    if (tokens.length) {
      filtered = filtered.filter(job => {
        if (!job) return false;
        
        // Include all searchable fields
        const searchableFields = [
          job.title || '',
          job.department || '',
          job.location || '',
          job.course || '',
          job.job_type || '',
          job.description || '',
          job.requirements || ''
        ].map(normalize);

        // For each token, require it to match at least one field (AND across tokens)
        return tokens.every(token => 
          searchableFields.some(field => field.includes(token))
        );
      });
    }

    // Sorting
    const sorted = [...filtered].sort((a, b) => {
      if (!a || !b) return 0;
      
      // Helper to parse salary strings into numbers (handles formats like "$80k - $100k" or "80000-100000")
      const parseSalary = (salaryStr) => {
        if (!salaryStr) return 0;
        // Handle ranges (take the average)
        const range = salaryStr.split('-').map(s => {
          // Remove non-numeric characters and convert to number
          const num = parseFloat(s.replace(/[^0-9.]/g, ''));
          // If the number is in thousands (e.g., 80k), convert to actual number
          return s.toLowerCase().includes('k') ? num * 1000 : num;
        });
        return range.reduce((sum, num) => sum + num, 0) / range.length;
      };

      switch (sortOption) {
        case 'newest':
          return new Date(b.postedAt) - new Date(a.postedAt);
        case 'oldest':
          return new Date(a.postedAt) - new Date(b.postedAt);
        case 'salary-high': {
          const salaryA = parseSalary(a.salary);
          const salaryB = parseSalary(b.salary);
          return salaryB - salaryA;
        }
        case 'salary-low': {
          const salaryA = parseSalary(a.salary);
          const salaryB = parseSalary(b.salary);
          return salaryA - salaryB;
        }
        case 'deadline':
          return new Date(a.deadline) - new Date(b.deadline);
        default:
          return new Date(b.postedAt) - new Date(a.postedAt);
      }
    });

    return sorted;
  }, [sortOption, debouncedQuery, jobs]);

  const paginatedJobs = useMemo(() => {
    const startIndex = (currentPage - 1) * pageSize;
    return filteredJobs.slice(startIndex, startIndex + pageSize);
  }, [filteredJobs, currentPage, pageSize]);

  const totalPages = Math.ceil(filteredJobs.length / pageSize);

  if (isLoading) {
    return (
      <Container className="d-flex justify-content-center align-items-center" style={{ minHeight: '50vh' }}>
        <Spinner animation="border" role="status">
          <span className="visually-hidden">Loading jobs...</span>
        </Spinner>
      </Container>
    );
  }

  if (error) {
    return (
      <Container className="py-5">
        <Alert variant="danger">
          <div className="d-flex align-items-center">
            <FaExclamationCircle className="me-2" />
            {error}
          </div>
        </Alert>
        <div className="text-center mt-3">
          <Button variant="primary" onClick={() => window.location.reload()}>
            Retry
          </Button>
        </div>
      </Container>
    );
  }

  return (
    <Container className="job-opportunities py-4">
      <div className="text-center mb-5">
        <h1 className="display-5 fw-bold mb-3">Find Your Opportunity Here!</h1>
        <p className="lead text-muted">Browse and apply to faculty positions from top institutions</p>

        <div className="search-container mx-auto mt-4" style={{ maxWidth: '800px' }}>
          <Form onSubmit={(e) => { e.preventDefault(); }}>
            <div className="d-flex flex-column flex-md-row gap-3">
              <div className="input-group flex-grow-1">
                <span className="input-group-text bg-white border-end-0">
                  <FaSearch className="text-muted my-2" />
                </span>
                <Form.Control
                  type="text"
                  placeholder="Search by job title, department, or location"
                  className="border-start-0"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                />
                <Button variant="primary" className="px-4" type="submit">Search</Button>
              </div>

              <div className="d-flex align-items-center">
                <div className="input-group" style={{ minWidth: '200px' }}>
                  <Form.Select
                    value={sortOption}
                    onChange={(e) => setSortOption(e.target.value)}
                    className="form-select border-start-0"
                    style={{ cursor: 'pointer' }}
                  >
                    <option value="newest">Newest First</option>
                    <option value="oldest">Oldest First</option>
                    <option value="salary-high">Salary: High to Low</option>
                    <option value="salary-low">Salary: Low to High</option>
                    <option value="deadline">Deadline</option>
                  </Form.Select>
                </div>
              </div>
            </div>
          </Form>
        </div>
      </div>

      <div>
        {showSaveAlert && lastSavedJob && (
          <Alert
            variant="success"
            className="d-flex align-items-center mb-4"
            onClose={() => setShowSaveAlert(false)}
            dismissible
          >
            <div className="d-flex align-items-center">
              <i className="bi bi-bookmark-check-fill me-2"></i>
              <span>Job saved: <strong>{lastSavedJob.title}</strong></span>
            </div>
            <Button
              variant="outline-success"
              size="sm"
              className="ms-5"
              onClick={() => {
                navigate('/faculty/saved-jobs');
                setShowSaveAlert(false);
              }}
            >
              View Saved
            </Button>
          </Alert>
        )}

        {filteredJobs.length === 0 ? (
          <div className="text-center py-5">
            <i className="bi bi-search text-muted" style={{ fontSize: '3rem', opacity: 0.5 }}></i>
            <h4 className="mt-3">No jobs found</h4>
            <p className="text-muted">Try adjusting your search criteria</p>
          </div>
        ) : (
          <Row className="g-4">
            {paginatedJobs.map((job) => (
              <Col key={job.id} xs={12}>
                <JobCard
                  job={job}
                  isSaved={isJobSaved(job.id)}
                  isApplied={appliedJobs.includes(job.id)}
                  onSaveToggle={handleSaveJob}
                  onApply={handleApply}
                />
              </Col>
            ))}
          </Row>
        )}

        {totalPages > 1 && (
          <div className="d-flex justify-content-center mt-5">
            <Pagination>
              <Pagination.Prev
                onClick={() => setCurrentPage(p => Math.max(1, p - 1))}
                disabled={currentPage === 1}
              />
              {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                let pageNum;
                if (totalPages <= 5) {
                  pageNum = i + 1;
                } else if (currentPage <= 3) {
                  pageNum = i + 1;
                } else if (currentPage >= totalPages - 2) {
                  pageNum = totalPages - 4 + i;
                } else {
                  pageNum = currentPage - 2 + i;
                }

                return (
                  <Pagination.Item
                    key={pageNum}
                    active={pageNum === currentPage}
                    onClick={() => setCurrentPage(pageNum)}
                  >
                    {pageNum}
                  </Pagination.Item>
                );
              })}
              <Pagination.Next
                onClick={() => setCurrentPage(p => Math.min(totalPages, p + 1))}
                disabled={currentPage === totalPages}
              />
            </Pagination>
          </div>
        )}
      </div>
    </Container>
  );
};

export default JobOpportunities;
